#!/usr/bin/env julia

using Pkg
Pkg.activate(".")

using VeluSqrt
using BenchmarkTools
using Primes
using Nemo
using EllipticCurves

#################### VeluResultant

println("# Benchmarking isogeny algorithm")

function find_prime(ell, bits=256)
    n = Int(round(bits - log2(ell) - 2))
    for cof in (BigInt(2)^n:BigInt(2)^(n+1))
        if Primes.isprime(4 * ell * cof - 1)
            return 4 * ell * cof - 1, 4 * cof
        end
    end
end

degrees = [nextprime(2^i) for i in (2:23)]

for (i, ell) in enumerate(degrees)
    p, cof = find_prime(ell, 512)
    F = GF(ZZ(p))
    E = Montgomery(F(0), F(1))
    P = XZzero(E)
    while isinfinity(P)
        P = cof * randXZ(E)
    end
    old = @benchmark VeluRenes($(P), $(ell), points) setup=(points=[randXZ($(E))]) gcsample=true evals=1
    new = @benchmark VeluResultant($(P), $(ell), points) setup=(points=[randXZ($(E))]) gcsample=true evals=1
    println("l ", ell, " ns ", join(new.times, " "), " baseline ", join(old.times, " "))
end

#################### CSIDH

println("# Benchmarking CSIDH")
csidh = CSIDH_setup()

KEYS, SAMPLES = 65, 7

for i in 1:KEYS
    key = CSIDH_sk(csidh)
    old = @benchmark CSIDH_action($(csidh), $(key); thresh=600) seconds=60 samples=SAMPLES evals=1 gcsample=true
    new = @benchmark CSIDH_action($(csidh), $(key); thresh=100) seconds=60 samples=SAMPLES evals=1 gcsample=true
    println("action_csidh ns ", join(new.times, " "), " baseline ", join(old.times, " "))
end


#################### CSURF

println("# Benchmarking CSURF")
csidh = CSURF_setup()

KEYS, SAMPLES = 65, 7

for i in 1:KEYS
    key = CSURF_sk(csidh)
    old = @benchmark CSIDH_action($(csidh), $(key); thresh=600) seconds=60 samples=SAMPLES evals=1 gcsample=true
    new = @benchmark CSIDH_action($(csidh), $(key); thresh=100) seconds=60 samples=SAMPLES evals=1 gcsample=true
    println("action_csurf ns ", join(new.times, " "), " baseline ", join(old.times, " "))
end


#################### B-SIDH

println("# Benchmarking B-SIDH")

E, p = BSIDH_params()

ker = generator(E, p, :minus)
P = generator(E, p, :plus)
Q = generator(E, p, :plus)
PQ = generator(E, p, :plus)

SAMPLES=5
old = @benchmark walk($(ker), $(p), :minus, [$(P), $(Q), $(PQ)], Inf) gcsample=true evals=1 samples=SAMPLES
new = @benchmark walk($(ker), $(p), :minus, [$(P), $(Q), $(PQ)], 100) gcsample=true evals=1 samples=SAMPLES
println("bsidh_p-1 ns ", join(new.times, " "), " baseline ", join(old.times, " "))

ker = generator(E, p, :plus)
P = generator(E, p, :minus)
Q = generator(E, p, :minus)
PQ = generator(E, p, :minus)

SAMPLES=3
old = @benchmark walk($(ker), $(p), :plus, [$(P), $(Q), $(PQ)], Inf) gcsample=true evals=1 samples=SAMPLES
new = @benchmark walk($(ker), $(p), :plus, [$(P), $(Q), $(PQ)], 100) gcsample=true evals=1 samples=SAMPLES
println("bsidh_p+1 ns ", join(new.times, " "), " baseline ", join(old.times, " "))
